!This programme simulates the thermal evolution of an asteroid heated by radioactive decay of short (Al-26, Fe-60) and long lived isopes (U-235,U-238,Th-232,K-40).
!This problem is treated as spherically symmetric where the heat conductivity equation is solved by a finite-difference method with a fixed temperatur at the surface.
!The model includes sintering according to either the theory of Rao/Kakar/Chacklader or Helle.

!The files of this programme are organised as follows. The main files, that are the ones to be compiled by "ifort filename" can be found in the main folder as this file, too.
!These files rely on other files found in the folder "Subroutinen" and are included by include directives. The output that a program produces is by default put into the folder "Ausgabe" (output).
!The names of files and modules in this programme package are often grown historically and thus do not always contain exactly what the names suggest.
!All names of these files and variables in the programme are generally in german (given with english translation in the comments, if the meaning is not apparent).

!Most options the user is supposed to change are in the first module "Modelleinstellungen" (model settings) while most model and nature constants can be found the in subroutine "Naturkonstanten" in the file "AsteroidKonstanten.h".
!At first the formation time ("Bildungszeit") describes when the body has formed respective to the event of first CAI-formation. 
!The formation of the body is in general assumed to be initially, although a growth subroutine is included to treat a linear mass growth.
!This formation time determines the amount of Al-26 still available at formation to heat up the body. The later the body forms the less hot it can become.
!Radius is the radius that the asteroid would have in the case of complete compacted material without porosity. If the material is porous the body will be larger.
!f_Fe60 is the amount of Fe-60 available at CAI-formation respective to the amount of Fe-56.
!The initial porosity (phi_0) denotes the porosity of the material the body forms from. This is only relevant if the logical "Gemisch" (mixture) is .false. (see below).
!The initial and surface temperature T_0 and T_S serve as initial and surface boundary condition. T_S represents the temperature of the environment where the body forms while the disk is still present ("Scheibe" = disk). 
!T_0 represents the mean temperature as a result of the equilibrium between irradiation from the sun and emmision from the asteroid surface after the disk has dispersed.
!The point in time when the disk disperses is set in the module "Naturkonstanten".
!Usually both temperatures where assumed to be the same during the model caluculations done by the author.
!The heat conductivity of fully compacted asteroid material at 300 K is given by the constant "Wlvf" (heat conductivity prefactor).
!This value is used to calculate a porosity and temperature dependent heat conductivity in the function "k_w_Mat_func" in the file "Funktionen.f90" in the folder "Subroutinen".

!If growth is considered, a small body of the size "Radius_Anfang" will begin to accrete mass starting at formation time. 
!The mass will be accreted with a linear mass accretion rate calculated from the given accretion duration ("Akkretionsdauer") and the final radius "Radius".

!The main material constant for the simulation of heating is the heat capacity. Two approaches for the heat capacity are supported in this program.
!With the logical cp_genau = .false. the fit to data of chondrites obtained by Yomogida & Matsui 1983 is used.
!With the logical cp_genau = .true. the heat conductivity is provided by a table given in file "Cp_H.dat" in Subroutinen.
!These data were obtained from the mineral composition of H chondrites. Other files for L chondrites (Cp_L.dat) and acapulcoites (Cp_A.dat) are also provided.

!Sintering in this programme can be treated by two ways.
!with Helle = .false. the hot isostatic pressing theory of Kakar & Chaklader 1967 and Rao & Chaklader 1972 is used.
!Alternatively  with Helle = .true. the theory of Helle et al. 1985 is used.

!In the latter case with "Gemisch" = .true (Gemisch = mixture) a two component material consisting of chondrules and matrix can be assumed (see below) otherwise only a one component material is considered.

!The spacial resolution k determines the number of mass shells that are used to simulate the parent body.

!The duration ("Dauer") determines for how long the time evolution of the thermal evolution is supposed to last. 
!Since all H chondrite closure ages are within 200 Ma after CAI formation, normally a duration of 200 Ma is sufficient.

!Finally if a two component mixture is assumed it is possible to determine the matrix volume partition in the matrix chondrule mixture.
!Here only values between 0.0 and 0.265 are allowed assuming a chondrule dominated material and values between 0.444 and 1.0 assuming a matrix dominated material.

!In the array "TieH" depths of nine layers can be defined where the temperature evolution is recorded during the model run and put out afterwards.

!The function/subroutine dependence of this program package is as follows. The location of the function/subroutine is always file "Funktionen.f90" if not stated otherwise in ():
!program Asteroid     					(Asteroid.f90)
!	-> subroutine Ast 						(Ast.f90)
!		-> function Kaltpressen
!		-> function Q_func
!		-> subroutine Fixpunktiteration	(Ast.f90)
!			-> function P_func
!			-> function Sintern or function SinternYom
!			-> function Diskretisierung
!				-> function k_w_Mat_func
!				-> function cp_func
!				-> function Tridia

!----------------------------------------------------------------------------------------------------

module Modelleinstellungen
	implicit none	

	!Asteroid model parameters
	real(kind=8) :: Bildungszeit  = 2d0 	!Formation time in Ma
	real(kind=8) :: Radius			  = 100d0	!Asteroid radius in km
	real(kind=8) :: f_Fe60 			  = 0.0115d0	!Fe-60/Fe-56-ratio in units of 10^-6
	real(kind=8) :: phi_0 			  = 0.3d0	!Initial porosity in parts of one
	real(kind=8) :: T_0 			    = 150d0	!Initial and surface boundary temperature in K after disk dispersal
	real(kind=8) :: T_S 			    = 150d0 !Initial and surface boundary temperature in K while disk is present	
	real(kind=8) :: Wlvf 			    = 4.9d0	!Bulk heat conductivity at 300 K in W/mK (used as based value which is modified by porosity and temperature later)
	real(kind=8) :: Radius_Anfang   = 10.d0	!Radius of Asteroid seed in km if asteroid growth is calculated
	real(kind=8) :: Akkretionsdauer = .0d0	!Duration of linear asteroid growth in Ma if asteroid growth is modelled
											
	logical :: cp_genau = .true. 	!.true.: heat conductivity from file used, .false.: heat conductivity calculated in function cp_func
	logical :: Helle    = .true.			!.true.: Sintering theory according to Helle used, .false.: Sintering according to Yomogida/Matsui used
	logical :: Gemisch  = .false.	!.true.: Initial material is two component material consisting of matrix and chondrules, 
											          !.false.: Initial material is one component material
	logical :: Einlesen = .true. 	!.true.: read in variables from file, .false.: use default values
	logical :: Para_Ausgabe = .false.
	integer :: k = 10			!Spacial resolution
	real(kind=8) :: Dauer = 200d6		!How many years thermal evolution is to proceed.
		
	!Only used if Gemisch = .true.
	real(kind=8):: f_ma = 0.15	!Matrix volume partition in matrix chondrule mixture. Only defined for 
								!values between 0.0 - 0.265 ("chondrendominiert") and 0.444 - 1.0 ("matrixdominiert")	
	
	!Depths of H chondrites in m: Estacado E6, Guarena G6, Kernouvé K6, Richardton R5, Allegan A5, 
	!Nadiabondi N5, Forest Vale F4, Ste. Marguerite S4, Mt. Brown M6
	!Abbreviations of the meteorites (as E6 for Estacado) are used frequently in this programme. 
	!The numbers denote the petrologic type.
	real(kind=8),dimension(9) :: TieH = (/ 40d0, 40d0, 30d0, 20d0, 10d0, 5d0, 3d0, 2d0, 1d0  /)
	
	integer,parameter :: ParZahl = 8		!Number of Parameters used for fitting in Evolution_H.f90
	logical :: matrixdominiert = .false.,chondrendominiert = .false. !do not change
	logical :: Wachstumsstop					
	logical, dimension(:),allocatable :: MaskeT			
	real(kind=8) :: Wachstumsgrenze = 0.01d6 !lower values of "Akkretionsdauer" than "Wachstumsgrenze" 
											                     !are considered to be 0 and will not trigger growing routine
	logical, dimension(:),allocatable :: D_schliessen	
end module Modelleinstellungen

!----------------------------------------------------------------------------------------------------

include 'Subroutinen/AsteroidKonstanten.h'

!----------------------------------------------------------------------------------------------------

!This module provides some constants and variables that are used everywhere
module Parameter
	use Naturkonstanten
	implicit none
	real(kind=8), dimension(:), allocatable :: Z_Vektor,Schalen	
	real(kind=8), dimension(:,:), allocatable :: T_Matrix
	real(kind=8) :: T__max
	real(kind=8) :: Radius_aus
	integer :: Lauf	,i_cp
	character(len=50) :: Bezeichnung, Ziel,ZielZL
  real(kind=8), dimension(:,:),allocatable :: Cp_Mat
  real(kind=8),dimension(:,:),allocatable :: STemp
	real(kind=8),dimension(:),allocatable :: STemp_Fehler
	real(kind=8),dimension(:,:),allocatable :: AKZ, AKZ_Fehler
	real(kind=8),dimension(:),allocatable :: Tmin,Tmax,Tmin_Fehler,Tmax_Fehler
	real(kind=8) :: CAI_Alter, CAI_Fehler
	logical,dimension(:,:),allocatable :: Maske 	!needed for summing up of chi's. False values are not considered.
	real(kind=8), dimension(2) :: BereichBZ,BereichRad,BereichFe60,BereichPor,BereichT_0,BereichWlvf,BereichAkk,BereichT_S 
  contains
  subroutine Werte
  	!For allocation instead of knowing size at begin
  	implicit none
  	allocate(Cp_Mat(cp_Zahl,4))
  end subroutine Werte
end module Parameter

!----------------------------------------------------------------------------------------------------

module Funktionen
	implicit none
	contains
	include 'Subroutinen/Funktionen.f90'
end module Funktionen

!----------------------------------------------------------------------------------------------------

!The main programme. This is basically just a starting routine for calling the subroutine "Ast" (to be found in file "Ast.f90" 
!which is included at the end of this file) which calculates the thermal evolution of asteroids.
program Asteroid
use Modelleinstellungen
use Naturkonstanten
use Parameter
use Funktionen
implicit none
	real(kind=8),dimension(8) :: Parameter_ein
	integer :: Lang,i
	real(kind=8), dimension(:), allocatable :: Radien,Zeiten
	real(kind=8), dimension(:,:), allocatable :: Temperaturen 
	interface
		subroutine Ast(Parameter_ein,Lang,Radien,Temperaturen,Zeiten)
			use Funktionen
			use Naturkonstanten
			use Modelleinstellungen
			use Parameter
			use omp_lib
			implicit none
			integer,intent(out) :: Lang
			real(kind=8), dimension(:),intent(out), allocatable :: Radien,Zeiten
			real(kind=8), dimension(:,:),intent(out), allocatable :: Temperaturen
			real(kind=8), dimension(ParZahl),intent(in) :: Parameter_ein
		end subroutine Ast
	end interface	
	
	if(Einlesen .eqv. .true.) then
		call Dateieingabe		
		if(Para_Ausgabe .eqv. .true.) stop
	end if

	!Defines programme type to be a calculation of one thermal evolution. 
	!Changing of Programmtyp in "Evolution" will trigger genetical algorithm. Not recommended here.	
	Programmtyp = 'Durchlauf'
	
	call Wertesetzen		!sets values in AsteroidKonstanten.h
	call Werte
	
	!Only used if gemisch = .true.
	if (f_ma <= 0.265d0 .and. f_ma >= 0.d0 .and. Gemisch .eqv. .true.) chondrendominiert = .true.
	if (f_ma >= 0.444d0 .and. f_ma <= 1.d0 .and. Gemisch .eqv. .true.) matrixdominiert = .true.
	
	if(Gemisch .eqv. .true.) then
		if ((chondrendominiert .eqv. .false.) .and. (matrixdominiert .eqv. .false.)) then
			write(*,*) "Error: f_ma has unvalid value"
			stop
		end if
	end if

	!Preparing for subroutine Ast
	!The variable Parameter_ein contains all the model parameters of the parent body and is an input variable for subroutine "Ast".
	Parameter_ein(1) = Bildungszeit;
	Parameter_ein(2) = Radius		
	Parameter_ein(3) = f_Fe60							
	Parameter_ein(4) = phi_0
	Parameter_ein(5) = T_0			
	Parameter_ein(6) = Wlvf
	Parameter_ein(7) = Akkretionsdauer							
	Parameter_ein(8) = T_S
	
	!Read in heat capacity values from file		
	open(97, file = 'Subroutinen/Cp_H.dat', access = 'sequential')
	do i_cp=1,Cp_Zahl
		read(97,*) Cp_Mat(i_cp,1),Cp_Mat(i_cp,2),Cp_Mat(i_cp,3),Cp_Mat(i_cp,4)
	end do
	close(97)	
		
	!Subroutine for thermal evolution in file Ast.f90. Output parameter "Lang" provides number of 
	!timesteps used.
	call Ast(Parameter_ein,Lang,Radien,Temperaturen,Zeiten)
	
	contains
	include "Subroutinen/Einlesen.f90"
end program Asteroid

!----------------------------------------------------------------------------------------------------

include "Subroutinen/Ast.f90"
