!This programme is using a genetical algorithm to fit the thermal evolution model to cooling ages of H chondrites.
!The cooling ages can be found in file "Alter_H.f90".
!The genetical algorithm used is the version 1.2 of Pikaia first published in Charbonneau 1995, ApJS 101,309. 
!The code for this was translated from Fortran 77 into Fortran 95 using "to_f90" 
!( Author - Alan Miller  (amiller @ bigpond.net.au) WWW-page: http://users.bigpond.net.au/amiller/.)
!This programme also supports parallel computing via openmp. The number of processor cores used is to 
!be controlled manually via the variable "Kerne" (Cores, kernels, nuclei; first one in module Parameter).

!The function/subroutine dependence of this program package is as follows. The location of the function/subroutine is always file 
!"Funktionen.f90" if not stated otherwise in ():
!program Evo 					(Evolution.f90)
!	-> subroutine Pikaia			(pikaia.f90)
!		-> function Opt_A			(Evolution.f90)			
!			-> subroutine Ast 						(Ast.f90)
!				-> function Kaltpressen
!				-> function Q_func
!				-> subroutine Fixpunktiteration	(Ast.f90)
!					-> function P_func
!					-> function Sintern or function SinternYom
!					-> function Diskretisierung
!						-> function k_w_Mat_func
!						-> function cp_func
!						-> function Tridia
!			-> function compare	(EvoRest.f90)

!----------------------------------------------------------------------------------------------------

module Modelleinstellungen
  !This module is widely identical to the same one in Asteroid_H.f90. 
  !A detailed commentation can be found there.
  implicit none
  logical :: cp_genau = .true.
  logical :: Helle = .true.
  logical :: Gemisch = .true.
  logical :: Einlesen = .true. 			!.true.: read in variables from file, .false.: use default values
  logical :: Para_Ausgabe = .false.
  integer :: k = 100
  real(kind=8) :: Dauer = 200d6

  real(kind=8) :: Radius
  real(kind=8) :: Bildungszeit
  real(kind=8) :: f_Fe60
  real(kind=8) :: T_0
  real(kind=8) :: T_S
  real(kind=8) :: phi_0
  real(kind=8) :: Wlvf
  real(kind=8) :: Akkretionsdauer
  !$omp threadprivate(Radius,Bildungszeit,f_Fe60,T_0,T_S,phi_0,Wlvf,Akkretionsdauer)
	
  !Only used if Gemisch = .true.
  real(kind=8) :: f_ma = 0.15d0	!Matrix volume partition in matrix chondrule mixture. Only defined
									              !for Values between 0.0-0.265 ("chondrendominiert") and 0.444-1.0 ("matrixdominiert")
									
  logical :: matrixdominiert = .false.,chondrendominiert = .false. !do not change
	
  logical :: Wachstumsstop
  logical, dimension(:),allocatable :: MaskeT
  real(kind=8) :: Wachstumsgrenze = 0.01d6
  real(kind=8) :: Radius_Anfang = 10.d3
  integer,parameter :: ParZahl = 8		!Number of Parameters used for fitting in Evolution_H.f90
  real(kind=8),dimension(3) :: TieAc
  real(kind=8),dimension(9) :: TieH
  !$omp threadprivate(Wachstumsstop,MaskeT,f_ma)

  logical, dimension(:),allocatable :: D_schliessen
  !$omp threadprivate(D_schliessen)
end module Modelleinstellungen

!----------------------------------------------------------------------------------------------------

include 'Subroutinen/AsteroidKonstanten.h'

!----------------------------------------------------------------------------------------------------

module Parameter
	use Naturkonstanten
	implicit none
	integer :: Kerne = 2 					!Numbers of cores to be used when run with openmp	
	
	integer,parameter :: Ergebnislange=30 	!Number of best fitting models kept in output file "EvoErgebnis"
	real(kind=8),dimension(:,:),allocatable :: STemp
	real(kind=8),dimension(:),allocatable :: STemp_Fehler
	integer :: Matdim !defines size of output matrix
	real(kind=8),dimension(:),allocatable :: chi_Tmin,Tmin,chi_Tmax,Tmax,Tmin_Fehler,Tmax_Fehler
	!$omp threadprivate(chi_Tmin,chi_Tmax)
	real(kind=8), dimension(:), allocatable :: Z_Vektor,Schalen
	real(kind=8), dimension(:,:), allocatable :: T_Matrix	
	real(kind=8) :: T__max
	!$omp threadprivate(Z_Vektor,T_Matrix,T__max,Schalen)
	real(kind=8),dimension(:,:),allocatable :: chi	
	!$omp threadprivate(chi)	
	logical,dimension(:,:),allocatable :: Maske 	!needed for summing up of chi's. False values are not considered.
	real(kind=8) :: chi_gesamt	
	!$omp threadprivate(chi_gesamt)
	real(kind=8), dimension(:,:),allocatable :: BesteZehnChi
	real(kind=8), dimension(:,:),allocatable :: BesteZehnChiTmax
	real(kind=8) :: Radius_aus	
	real(kind=8),dimension(:,:),allocatable :: AKZ, AKZ_Fehler
	real(kind=8) :: CAI_Alter, CAI_Fehler		
	integer :: Generation, ngen, Individuum	
	!$omp threadprivate(Individuum)
	integer :: AusgabeWert = 1
	integer :: Wert
	!$omp threadprivate(Wert)
	real(kind=8) :: instWachs = 9.d3
	real(kind=8), dimension(:,:),allocatable :: Cp_Mat
	character(len=50) :: Bezeichnung, Ziel,ZielZL
  integer :: Lenge, Lauf
  real(kind=8),dimension(:),allocatable :: T_max, Zeit_Max
	!$omp threadprivate(Lauf,T_max,Zeit_Max)
  real(kind=8), dimension(2) :: BereichBZ,BereichRad,BereichFe60,BereichPor,BereichT_0,BereichWlvf,BereichAkk,BereichT_S 
	contains
	subroutine Werte
  	implicit none
  	allocate(Cp_Mat(cp_Zahl,4))		
  end subroutine Werte	
end module Parameter

!----------------------------------------------------------------------------------------------------

module Funktionen
	implicit none
	contains
	include 'Subroutinen/Funktionen.f90'
end module Funktionen

!----------------------------------------------------------------------------------------------------

module Ubergabe
	use parameter
    REAL(kind=8),dimension(ParZahl) :: Parameter_ein
    real(kind=8),dimension(:),allocatable :: Schicht_ein
	!$omp threadprivate(Parameter_ein,Schicht_ein)
end module Ubergabe

!----------------------------------------------------------------------------------------------------

!This is the main programme of this file. It mainly calls the subroutine Pikaia found in file pikaia.f90.
!This subroutine has the function OptA as input which can be found in this file. 
!If you change the number of free parameters for the fitting procedure make sure you also change the variable "N" in the main 
!programme respectively.
program Evo
use Modelleinstellungen
use parameter
use Naturkonstanten
implicit none
	integer :: seed,status1,i_cp,i,j
	integer,parameter :: n=ParZahl 					!Choose here dimension of randomly generated number array
	real(kind=4) :: CTRL(12),x(n),F,OptA 	
	external OptA	

	seed = 1
	if(seed .eq. 0) GOTO 30 

	call rninit(seed) 
	! Set control variables (use defaults)                              
	do i=1,12 
		ctrl(i)=-1 
	end do 
	!!!!!!!!!!                                                              
	ctrl(12)=1 
	!!!!!!!!!! 
	Matdim=ParZahl+Metzahl+4
	allocate(BesteZehnChi(Matdim,Ergebnislange))
	allocate(BesteZehnChiTmax(MetZahl,Ergebnislange))
	allocate(Tmin(MetZahl))	
	allocate(Tmax(MetZahl))
	allocate(Tmin_Fehler(MetZahl))
	allocate(Tmax_Fehler(MetZahl))
	
	BesteZehnChi=0.d0
	BesteZehnChiTmax=0.d0
	do i=1,Ergebnislange
		BesteZehnChi(ParZahl+MetZahl+1,i) = 9999d0 
	end do
	
	!Bitte gib Ziel ohne / an.
	Ziel = 'Ausgabe'      			!Folder where data are put out. Standard is "Ausgabe" (output).                                                      
	Bezeichnung = 'Name'	      !Name that all output files have in common
	Lenge= len(trim(Ziel))+1
	Ziel(Lenge:Lenge) = '/'
	!Protokoll file lists results of every model calculated during optimisation
	open(100, file = trim(adjustl(Ziel))//"EvoProtokoll"//trim(adjustl(Bezeichnung))//".dat", status = 'replace')
	close(100)

  include 'Subroutinen/Alter_H.f90'
	call Wertesetzen
	call Werte 
	Programmtyp = 'Evolution'
		
	!Get input data from files
	if(Einlesen .eqv. .true.) then
		call Dateieingabe		
	end if

	!mixture specifics
	if (f_ma <= 0.265d0 .and. f_ma >= 0.d0 .and. Gemisch .eqv. .true.) chondrendominiert = .true.	
	if (f_ma >= 0.444d0 .and. f_ma <= 1.d0 .and. Gemisch .eqv. .true.) matrixdominiert = .true.
	if ((chondrendominiert .eqv. .false.) .and. (matrixdominiert .eqv. .false.) .and. (Gemisch .eqv. .true.)) then	
		write(*,*) "Error: f_ma has unvalid value"
		stop
	end if
	
	!Get heat capacity data from file
	open(97, file = 'Subroutinen/Cp_H.dat', access = 'sequential')
	do i_cp=1,Cp_Zahl
		read(97,*) Cp_Mat(i_cp,1),Cp_Mat(i_cp,2),Cp_Mat(i_cp,3),Cp_Mat(i_cp,4)
	end do
	close(97)	  
	
	!This calls the genetical algorithm Pikaia which uses the function OptA to be found next in this file. 
	!Genetic algorithm for optimization (Charbonneau 1995, ApJS 101,309)                   
	call Pikaia(OptA,N,CTRL,X,F,status1)

	!Print the results                                                 
	WRITE(*,*) ' status: ',status1 
	WRITE(*,*) '      x: ',X 
	WRITE(*,*) '      f: ',F 
	WRITE(*,20) CTRL 
	20 FORMAT('    ctrl: ',6F11.6,/,10X,6F11.6) 
	write(*,*) "Programme executed succesfully"
	30 WRITE(*,31) 
	31 FORMAT('  --- program aborted ---') 
	
	contains
	include "Subroutinen/Einlesen.f90"
end program Evo 

!----------------------------------------------------------------------------------------------------

!This function is the optimisation function for the genetic algorithm. It first calls subroutine Ast to calculate the thermal 
!evolution of the asteroid.
!Then it calls the function "compare" (to be found in file "EvoRest.f90") which calculates the chi squared value for the 
!calculated thermal evolution model. 
!Then it manages the output into the output files and finally returs chi squared to pikaia.                      
real(kind=4) function OptA(n,x)
	use Parameter
	use Ubergabe
	use Modelleinstellungen
	use omp_lib
	implicit none      
	integer n ,i,j,Lang,jott
	real(kind=4) :: x(n)
	interface
		function Compare(Lang,xmin,Radien,Temperaturen,Zeiten)
			use Parameter, only :MetZahl
			implicit none
			real(kind=8), intent(inout),dimension(MetZahl) :: xmin
			real(kind=4),dimension(MetZahl) :: compare
			integer, intent(in) :: Lang
			real(kind=8), dimension(:),intent(in), allocatable :: Radien,Zeiten
			real(kind=8), dimension(:,:),intent(in), allocatable :: Temperaturen 			
		end function Compare	
		subroutine Ast(Parameter_ein,Lang,Radien,Temperaturen,Zeiten)
			use Funktionen
			use Naturkonstanten
			use Modelleinstellungen
			use Parameter
			use omp_lib
			implicit none
			integer,intent(out) :: Lang
			real(kind=8), dimension(:),intent(out), allocatable :: Radien,Zeiten
			real(kind=8), dimension(:,:),intent(out), allocatable :: Temperaturen
			real(kind=8), dimension(ParZahl),intent(in) :: Parameter_ein
		end subroutine Ast
	end interface 
	real(kind=8), dimension(MetZahl) :: RadPos ,Tiefe
	real(kind=4), dimension(MetZahl) :: schnitt
	real(kind=8), dimension(MetZahl) :: xmin
	integer, dimension(MetZahl) :: RadPosInt
	real(kind=8), dimension(:), allocatable :: Radien,Zeiten
	real(kind=8), dimension(:,:), allocatable :: Temperaturen
		
	
	allocate(Schicht_ein(MetZahl))
	allocate(Schalen(0:k))	
	allocate(chi_Tmin(MetZahl))
	allocate(chi_Tmax(MetZahl))	
	allocate(T_max(MetZahl))
	allocate(Zeit_max(MetZahl))
  allocate(chi(MetZahl,SchliessTempZahl)) 
  
	!This defines the model parameters of the asteroid which are used in the subroutine Ast.
	!You can choose an appropriate parameter space for each parameter as you like.
	!The array x of dimension N is randomly generated by a random number generator.
	!The dimension of N must be defined in the main program (program Evo). 	
	Parameter_ein = 0.d0
	Parameter_ein(1) = BereichBZ(1) + (BereichBZ(2)-BereichBZ(1))*x(1)    !1.5d0+1.d0 * x(1)				!Formation time in Ma after CAI-formation
	Parameter_ein(2) = BereichRad(1) + (BereichRad(2)-BereichRad(1))*x(2)    !100.d0 + 230.d0 * x(2)			!Radius in km
	Parameter_ein(3) = BereichFe60(1) + (BereichFe60(2)-BereichFe60(1))*x(3)    !0.0115d0							!Fe60/Fe56-ratio in 10^-6
	Parameter_ein(4) = BereichPor(1) + (BereichPor(2)-BereichPor(1))*x(4)    !0.3d0							!Porosity in fraction of one
	Parameter_ein(5) = BereichT_0(1) + (BereichT_0(2)-BereichT_0(1))*x(5)    !150.d0+250.d0*x(5)				!initial and surface temperature after disk disposal in K
	Parameter_ein(6) = BereichWlvf(1) + (BereichWlvf(2)-BereichWlvf(1))*x(6)    !1.d0 + 4.d0 * x(6)				!bulk heat conductivity
	Parameter_ein(7) = BereichAkk(1) + (BereichAkk(2)-BereichAkk(1))*x(7)    !0d6 								!accretion duration
	Parameter_ein(8) = Parameter_ein(5)		!BereichT_S(1) + (BereichT_S(2)-BereichT_S(1))*x(8) 			!Initial and surface temperature before disk dispersal

	!For showing correct porosity in output files
	if (Gemisch .eqv. .true.) then	
		if (matrixdominiert .eqv. .true.)   Parameter_ein(4) = 1.d0 - D_cp / ( 1.d0-(1.d0-D_cp)*(1.d0-f_ma) ) 
		if (chondrendominiert .eqv. .true.) then
			f_ma = 0.15d0 
			Parameter_ein(4) = 1.d0 - D_cp / (1.d0-f_ma)
		end if
	end if

	wert = 0;	Lauf = 0
	
	!Call of thermal asteroid evolution subroutine
	call Ast(Parameter_ein,Lang,Radien,Temperaturen,Zeiten)
	
	!Evaluation of the model quality of the calculated thermal evolution
	schnitt = compare(Lang,xmin,Radien,Temperaturen,Zeiten)

	RadPos = xmin* Parameter_ein(2)
	RadPosInt = int(RadPos)
	Tiefe = 	Schalen(k)-RadPos

	!Output generation. Has to be in openmp critical block.
	!$omp critical
		call Ausgabe
	!$omp end critical
	  
	wert = 1      

  !output chi²
	OptA=-SQRT(ABS(sum(schnitt))) 

	deallocate(Schalen)
	deallocate(Schicht_ein)	
	deallocate(chi_Tmin)
	deallocate(chi_Tmax)	
	deallocate(T_max)
	deallocate(Zeit_max)
	deallocate(chi) 
	
	contains
		subroutine Ausgabe
			implicit none
				!Output to terminal
				write(*,'(A10,x,I4)') "Generation", generation
		
				!Parameters
				do i=1,ParZahl-1
					write(*,'(A10,x)',advance="no") Parametername(i)
				end do
				write(*,'(A10,x)') Parametername(ParZahl)		
				do i=1,ParZahl-1
					write(*,'(F10.4,x)',advance="no") Parameter_ein(i)
				end do
				write(*,'(F10.4,2x)') Parameter_ein(ParZahl)		

				!Meteorites
				do i=1,MetZahl-1
					write(*,'(A12)',advance="no") adjustr(Meteoritenname(i))
				end do
				write(*,'(A12)') adjustr(Meteoritenname(MetZahl))
				do i = 1,MetZahl-1
					write(*,'((F12.4))',advance="no") Tiefe(i)
				end do 			
				write(*,'((F10.4))') Tiefe(MetZahl)
		
				write(*,'(A16,x,F9.2)') "Max. temperature", T__max

				!Model quality contributions from all data points each sorted by Meteorite and decay system
				write(*,'(A4,x)',advance="no") 	"Type"!,				"Hf-W"	,"U-Pb-Pb","Ar-Ar","PuFT",	"Pb-Pb", "Al-Mg", "Tmax"
				do i=1,SchliessTempZahl
					write(*,'(A10,x)',advance="no") adjustr(Schliesstempname(i))
				end do
				write(*,'(A10)') "Tmax"
				do i=1,MetZahl
					write(*,'(A4,x)',advance="no") 	adjustl(Meteoritenname(i))!, chi(i,1),chi(i,2),chi(i,3),chi(i,4),chi(i,5),chi(i,6),chi_Tmax(i)
					do j=1,SchliessTempZahl
						write(*,'(F10.5,x)',advance="no") chi(i,j)
					end do
					write(*,'(F10.5,x)') chi_Tmax(i)
				end do

				!Total model quality
				write(*,'(2(A4,2x,F12.5,2x))') "chi ", sum(schnitt)
				write(*,*) " "

				!Filling the matrix that is to put out in file "EvoErgebnis"					
				if (sum(schnitt) < BesteZehnChi(MetZahl+ParZahl+1,1)) then
					do i = Ergebnislange,2,-1
						do jott=1,MetZahl+ParZahl+3
							BesteZehnChi(jott,i) = BesteZehnChi(jott,i-1)
						end do
						do jott=1,MetZahl
							BesteZehnChiTmax(jott,i) = BesteZehnChiTmax(jott,i-1)
						end do
					end do
					do i = 1,ParZahl
						BesteZehnChi(i,1) = Parameter_ein(i)
					end do	
					do i=1,MetZahl
						BesteZehnchi(ParZahl+i,1) = Tiefe(i)
						BesteZehnchiTmax(i,1) = T_Max(i)
					end do
					BesteZehnchi(MetZahl+ParZahl+1,1)= sum(schnitt)
					BesteZehnchi(MetZahl+ParZahl+2,1)= Generation
					BesteZehnchi(MetZahl+ParZahl+3,1)= T__Max
				end if

        !Output in EvoResult
				open(99, file = trim(adjustl(Ziel))//"EvoResult"//trim(adjustl(Bezeichnung))//".dat", status = 'replace')
		
				do i=1,ParZahl
					write(99,'(A11)',advance="no") adjustr(Parametername(i))
				end do
				do i=1,MetZahl
					write(99,'(A12)',advance="no") adjustr(Meteoritenname(i))
				end do
				write(99,'(A10,A4,A10)') "chi","Gen","T_Max(0)"
				
				do i=1,Ergebnislange
					do j=1,ParZahl		
						write(99,'(F11.5)',advance="no") BesteZehnChi(j,i)
					end do
					do j=1,MetZahl
						write(99,'(F12.6)',advance="no") BesteZehnChi(ParZahl+j,i)
					end do
					write(99,'(F10.4)',advance="no") BesteZehnChi(ParZahl+MetZahl+1,i)
					write(99,'(I4)',advance="no") int(BesteZehnChi(ParZahl+MetZahl+2,i))
					write(99,'(F10.4)') BesteZehnChi(ParZahl+MetZahl+3,i)

					do j=1,ParZahl
						write(99,'(11x)',advance="no")
					end do
					do j=1,MetZahl-1	
						write(99,'(F12.2)',advance="no") BesteZehnchiTmax(j,i)
					end do
					write(99,'(F12.2)') BesteZehnchiTmax(MetZahl,i)
					write(99,*) " "		
				end do
				close(99)

				!Output in EvoProtokoll
				open(100, file = trim(adjustl(Ziel))//"EvoProtokoll"//trim(adjustl(Bezeichnung))//".dat",  status='old', &
					&action='write',position='append')
				do i = 1,ParZahl
						write(100,'(F14.6,x)',advance="no") Parameter_ein(i)
				end do
				do i=1,MetZahl
					write(100,'(F10.5,x)',advance="no") Tiefe(i)
				end do
				write(100,'(F12.4,x)',advance="no") sum(schnitt)
				write(100,'(I6)') Generation	
				close(100)
		end subroutine Ausgabe
end function OptA

!----------------------------------------------------------------------------------------------------

include 'Subroutinen/EvoRest.f90'
