!Calculation of heat capacity cp which is used in function "Diskretisierung"
!Either uses a fit from Yomogida/Matsui or interpolates in table Cp_Mat from file Cp_H.dat
!Input parameters are temperature T, heat capacity table Cp_Mat, and the position j_alt within the 
!heat capacity table at the last timestep to abbreviate the time needed for searching within the table
function cp_func(T,Cp_Mat,j_alt)
	use Naturkonstanten
	use Modelleinstellungen
	implicit none
	real(kind=8),dimension(:),allocatable,intent(in) :: T	
	integer :: i,j, indexx
	real(kind=8), dimension(cp_Zahl,4),intent(in) :: Cp_Mat 
	real(kind=8),dimension(:),allocatable :: cp_func	
	integer,dimension(:),allocatable,intent(inout) :: j_alt
	
	allocate(cp_func(0:k))
	
	if (cp_genau .eqv. .true.) then		!Checks if cp calculated by fit or from table	
		!Interpolation from values in "Cp_H.dat"	
		do i = 0,k		
			j=j_alt(i) - 10
			if (j < 3) j = 3
			!Cp_Mat(j,1) is a given temperature
			!Cp_Mat(j,2) is heat capacity for that temperature
			indexx = j
			if(T(i) > 5000.d0 ) write(*,*) "cp_func() Temperatur zu hoch", T(i)		!Tabled values 
															!only valid until a certain temperature
			do while( Cp_Mat(j,1) < T(i) )
				indexx = j
				j=j+1
			end do
			
			Cp_func(i) = Cp_Mat(indexx,2) + (Cp_Mat(indexx+1,2)-Cp_Mat(indexx,2)) * (T(i)-Cp_Mat(indexx,1)) &
				&/ ( Cp_Mat(indexx+1,1)-Cp_Mat(indexx-1,1) )
			j_alt(i) = j			
		end do
	else
		!Fit from Yomogida/Matsui
		do i = 0,k
			if ( T(i) >= 300.d0) then 
				!Fit from Yomogida/Matsui for temperatures of more than 300 K
				Cp_func(i) = cp_0 + 0.25d0*T(i) - 1.5d7/T(i)**2 	
			else
				!Low temperature heat conductivity by Debye-law
				Cp_func(i)=5.94456d0/(1.d0/(59.9716d0*T(i)**0.157916d0)**0.75d0+1.d0/(5.61815d-05*T(i)**3)**0.75d0)**(4.d0/3.d0) 
			end if	
		end do
	end if	
end function cp_func

!-----------------------------------------------------------------------------------------------------

!Calculation of heat conductivity k_w
!Input: temperature T, filling density D, pressure P, number of mass shell of surface layer j_alt
function k_w_Mat_func(T,D,P,k_ein)
	use Naturkonstanten
	use Modelleinstellungen
	implicit none
	integer :: i
	real(kind=8),dimension(:),allocatable,intent(in) :: T,D,P
	real(kind=8), dimension(:),allocatable :: k_t, Strahlung
	real(kind=8),dimension(0:k) :: k_w_Mat_func
	integer,intent(in) :: k_ein
	
	allocate(k_t(0:k))
	allocate(Strahlung(0:k))
	
	k_w_Mat_func = 0.d0
	do i=0,k_ein
		!Temperature dependence of heat conductivity according to Xu et al.
		k_t(i) = Wlvf * sqrt(298.d0/T(i))*(1.d0+0.032d0*P(i)/1.d9)	
	
		!Contribution from radiative heat transfer
		Strahlung(i) = 	(7.46d0-17.7d0*(1.d0-D(i)) + 11.2d0**(1.d0-D(i))**2)*1.d-3 + &
						& T(i)**3 * (1.93d0-5.18d0*(1.d0-D(i))+3.81*(1.d0-D(i))**2) * 1.d-10
		
		!Porosity dependence by fit to data of H/L-Chondrites from Henke et al., 
		!Data from Yomogida/Matsui
		k_w_Mat_func(i) = k_t(i) * (exp(-4.4d0-4.d0*(1.d0-D(i))/0.167d0) + exp(-4.d0*(1.d0-D(i))/0.08d0) )**(0.25d0) &
		& + 1.4d-2*P(i)/1.d5*(1.d0-D(i))**(2.d0/3.d0) + Strahlung(i)		
	end do		
end function k_w_Mat_func

!-----------------------------------------------------------------------------------------------------

!Calculation of pressure within the asteroid
!Input: filling density D, masses M, temperature T, pressure P
!Output: density rho, radius r, gas pressure P_G 
function P_func(D,M,rho,r,P_G,T,k_ein)
	use Naturkonstanten
	use Modelleinstellungen
	implicit none	
	real(kind=8) :: dMi, dMim, dPi, dPim,dPi_g,dPim_g
	real(kind=8), dimension(:), allocatable, intent(in) :: D,M,T		
	real(kind=8), dimension(:), allocatable, intent(out) :: rho,r,P_G
	real(kind=8), dimension(:), allocatable :: P_func, P_test, P_testv
	integer :: i
	integer,intent(in) :: k_ein	
	
	allocate(rho(0:k))
	allocate(P_test(0:k))
	allocate(P_testv(0:k))
	allocate(P_func(0:k))
	allocate(R(0:k))
	allocate(P_G(0:k))
	
	rho=0.d0
	r=0.d0
	P_G=0.d0

	!Arbitrary initialisation of test variables for triggering iteration loop for the first time.
	P_testv = 3.d0
	P_test  = 2.d0	
	do while ( maxval(abs(P_testv-P_test)/(P_test+1.d-15)) > 1.d-12 )			
		P_testv = P_test 
			
		!Density calculation by multiplying initial given global material density rho_0 with filling density D.
		do i=0,k_ein,1
			if (matrixdominiert .eqv. .true.) then			
				rho(i) = D(i) * ( (1.d0-f_ma)*rho_C + f_ma*rho_M ) / ( f_ma+(1.d0-f_ma)*D(i) )			
			else
				rho(i) = rho_0 * D(i)					
			end if
		end do	
		
		!Radii calculation from mass shells using the aforecalculated density rho
		do i=1,k_ein		
			dMi = M(i)-M(i-1)			
			r(i) = ( dMi/(vpid*(rho(i))) + r(i-1)**3 )**(drittel)				
		end do	

		!Pressure calculation		
		P_func(k_ein) = 0.d0		
		dMi = M(k_ein-1)-M(k_ein)			
		dPi = -Gr*M(k_ein)/(4.d0*pi*r(k_ein)**4)		
		P_func(k_ein-1) = P_func(k_ein) + dMi * dPi
		
		P_g(k_ein) = P_C
		dPi_g = -Gr*M(k_ein)/(4.d0*pi*r(k_ein)**4) * sqrt(k_B*T(k_ein)/mu_g) * P_g(k_ein)		
		P_g(k_ein-1) = P_g(k_ein) + dMi * dPi_g	
					
		do i = k_ein-1,2, -1
			dMi = M(i-1)-M(i)
			dMim = M(i)-M(i+1)
			!solid matter
			dPi = -Gr*M(i)/(4.d0*pi*r(i)**4)
			dPim = -Gr*M(i+1)/(4.d0*pi*r(i+1)**4)
			P_func(i-1) = P_func(i) + dMi * dPi + 0.5d0 * dMi**2 * (dPi-dPim)/dMim

			!Gas in Pores
			dPi_g = -Gr*M(i)/(4.d0*pi*r(i)**4) * sqrt(k_B*T(i)/mu_g) * P_g(i)
			dPim_g = -Gr*M(i+1)/(4.d0*pi*r(i+1)**4) * sqrt(k_B*T(i+1)/mu_g) * P_g(i+1)
			P_g(i-1) = P_g(i) + dMi * dPi_g + 0.5d0 * dMi**2 * (dPi_g-dPim_G)/dMim						
		end do		
		dMi = M(0)-M(1)
		dPi = -Gr*M(1)/(4.d0*pi*r(1)**4)
		dPi_g = -Gr*M(1)/(4.d0*pi*r(1)**4) * sqrt(k_B*T(1)/mu_g) * P_g(1)
		P_g(0) = P_g(1) + dMi * dPi_g		
		P_func(0) = P_func(1) + dMi * dPi						
		P_test = P_func			
	end do	
end function P_func

!-----------------------------------------------------------------------------------------------------

!Calculation of cold pressing, only used once and only if initial porosity larger than 37 %
!Input: filling density D, masses M, pressure P, temperature T, spacial resolution k_ein
!Output: filling density D, density rho, gas pressure P_G, radius r
function Kaltpressen(D,M,rho,r,P_G,T,k_ein)
	use Modelleinstellungen
	use Naturkonstanten
	implicit none
	integer,intent(in) :: k_ein
	integer :: i
	real(kind=8) :: D_0
	real(kind=8), dimension(:),allocatable :: Dtest, D1, Kaltpressen
	real(kind=8), dimension(:), allocatable, intent(in) :: M,T		
	real(kind=8), dimension(:), allocatable, intent(out) :: rho,r,P_G
	real(kind=8), dimension(:), allocatable, intent(inout) :: D	
	
	allocate(Dtest(0:k_ein))
	allocate(D1(0:k_ein))
	allocate(Kaltpressen(0:k_ein))	

	Kaltpressen = P_func(D,M,rho,r,P_G,T,k_ein)
	Dtest = D
	D_0 = 1.d0-phi_0

	do i=0, k_ein
		if (D(i) >= D_0 .and. D(i) <= 0.63d0) then
			D(i) = min(max(0.65d0 - 0.46d0 / ( (Kaltpressen(i)/0.13d5 )**(1.72d0) + 1.d0 ),D_0),0.63d0)
		end if	
	end do
	
	Kaltpressen = P_func(D,M,rho,r,P_G,T,k_ein)
	
	do while (maxval(abs(Dtest - D)/D) > 1.d-12)
		Dtest = D
		do i=0, k_ein
			if (D(i) >=D_0 .and. D(i) <= 0.63d0) D(i) = &
				&min(max(0.65d0 - 0.46d0 / ( (Kaltpressen(i)/0.13d5 )**(1.72d0) + 1.d0 ),D_0),0.63d0)	
		end do				
		Kaltpressen = P_func(D,M,rho,r,P_G,T,k_ein)	
	end do
end function Kaltpressen

!-----------------------------------------------------------------------------------------------------

!Calculation of heat production in shell i per kg. Output of produced heat per year in J/(kg*a)
!Input: time Zeit
function Q_func(Zeit)
	use Naturkonstanten
	use Modelleinstellungen
	implicit none 
	real(kind=8) :: Zei
	real(kind=8), intent(in) :: Zeit
	real(kind=8), dimension(0:k) :: Q_func
	real(kind=8) :: Konstante, h_Al, h_Fe, h_Long
	real(kind=8) :: Q_Al26, Q_U238, Q_U235, Q_Th232, Q_K40, Q_Fe60	
	
	Zei = Zeit
	Konstante = e_El/u_At
	Q_Al26 = Konstante * X_Al / u_Al26 * f_Al26 * E_Al26 * log(2.d0) / (T_Al26)
	Q_U238 = Konstante * X_U / u_U238 * f_U238 * E_U238 * log(2.d0) / (T_U238)
	Q_U235 = Konstante * X_U / u_U235 * f_U235 * E_U235 * log(2.d0) / (T_U235) 
	Q_Th232 = Konstante * X_Th / u_Th232 * f_Th232 * E_Th232 * log(2.d0) / (T_Th232)
	Q_K40 = Konstante * X_K / u_K40 * f_K40 * E_K40 * log(2.d0) / (T_K40)
	Q_Fe60 = Konstante * X_Fe / u_Fe60 * f_Fe60 * E_Fe60 * log(2.d0) / (T_Fe60)		
	h_Al   = Q_Al26*exp(-log(2.d0)/T_Al26*Zei)
	h_Fe   = Q_Fe60*exp(-log(2.d0)/T_Fe60*Zei)
	h_Long = (Q_U238*exp(-log(2.d0)/T_U238*Zei) + Q_U235*exp(-log(2.d0)/T_U235*Zei) + &
		& Q_Th232*exp(-log(2.d0)/T_Th232*Zei) + Q_K40*exp(-log(2.d0)/T_K40*Zei) )		
	Q_func = (h_Al+h_Fe+h_Long)
end function Q_func

!-----------------------------------------------------------------------------------------------------

!Solves heat conduction equation
!Input: time Zeit, time step size dt, heat capacity table Cp_Mat, filling density of new time step Dp, 
!pressure of new time step Pp, temperature T, temperature of new time step Tp, 
!density of new time step rho_p, masses M, heat development Q_ein, radii of new timestep Rp
function Diskretisierung(rho_p,T,Tp,Dp,Pp,M,Zeit,dt,Q_ein,Cp_Mat,Rp,j_alt,k_ein)
	use Naturkonstanten
	use Modelleinstellungen
	use omp_lib
	implicit none	
	real(kind=8), dimension(:),allocatable,intent(in) :: Dp,Pp,T,Tp,rho_p,M,Q_ein,Rp
	real(kind=8), intent(in) :: Zeit,dt
	real(kind=8), intent(in),dimension(cp_Zahl,4) :: Cp_Mat
	real(kind=8), dimension(:),allocatable :: Diskretisierung	
	integer,dimension(:), allocatable, intent(inout) :: j_alt
	integer,intent(in) :: k_ein
	integer :: i
	real(kind=8) :: F = 16.d0*pi**2
	real(kind=8) :: Theta = 0.d0, ThetaM	
	real(kind=8),dimension(:), allocatable :: A_r,A_l,X,Y,Z,R,hl,hr,cp_p,q,Br,Bl,kw,S,R_SFe,Y_SFe,R_SSil,Y_SSil
	
	allocate(A_r(0:k))
	allocate(A_l(0:k))
	allocate(X(0:k))
	allocate(Y(0:k))
	allocate(Z(0:k))
	allocate(R(0:k))
	allocate(hl(0:k))
	allocate(hr(0:k))
	allocate(cp_p(0:k))
	allocate(q(0:k))
	allocate(Diskretisierung(0:k))
	allocate(Br(0:k))
	allocate(Bl(0:k))
	allocate(kw(0:k))
	allocate(S(0:k))
	allocate(R_SFe(0:k))
	allocate(Y_SFe(0:k))
	allocate(R_SSil(0:k))
	allocate(Y_SSil(0:k))

	ThetaM = 1.d0-Theta

	hl=0.d0;	hr=0.d0
	A_l=0.d0;	A_r=0.d0
	Br=0.d0;	Bl=0.d0
	X=0.d0;		Y=0.d0;	Z=0.d0
	R=0.d0
	cp_p=0.d0
	q=0.d0
	Diskretisierung=0.d0
	kw=0.d0

	!spacial steps
	do i=1,k_ein-1	
		hr(i) = M(i+1)-M(i)
		hl(i) = M(i)-M(i-1)
	end do
	
	!Calculation of heat conductivity kw and heat capacity cp_p
	kw 	= spa* k_w_Mat_func(Tp,Dp,Pp,k_ein)
	cp_p= cp_func(Tp,Cp_Mat,j_alt)
	q 	= Q_ein/( cp_p ) 	

	do i=1,k_ein-1	
		A_r(i) = ( rho_p(i+1) * Rp(i+1)**4 * kw(i+1) + rho_p(i) * Rp(i)**4 * kw(i) ) * 0.5d0 	  
		A_l(i) = ( rho_p(i-1) * Rp(i-1)**4 * kw(i-1) + rho_p(i) * Rp(i)**4 * kw(i) ) * 0.5d0 	
		Br(i) = F*A_r(i)/(0.5d0*cp_p(i)*(hr(i)+hl(i))*hr(i)) 									
		Bl(i) = F*A_l(i)/(0.5d0*cp_p(i)*(hr(i)+hl(i))*hl(i)) 
		X(i) = 	  - dt * Br(i)				
		Z(i) = 	  - dt * Bl(i)					

		!Melting of iron phase
		R_SFe(i) = 0.d0
		Y_SFe(i) = 0.d0
		if ((Tp(i) >= Sol_Fe) .and. (Tp(i) <= Liq_Fe) .and. (T(i) >= Sol_Fe) .and. (T(i) <= Liq_Fe)) then		
			S(i) = Lw_Fe/ ( cp_p(i)*(Liq_Fe-Sol_Fe) ) * Teil_Fe
			Y_SFe(i) = +S(i) 		
			R_SFe(i) = - S(i) * T(i)					
		else if ((Tp(i) < Sol_Fe) .and. (T(i) < Sol_Fe)) then
		else if ((Tp(i) > Liq_Fe) .and. (T(i) > Liq_Fe)) then
		else if ((Tp(i) >= Sol_Fe) .and. (T(i) < Sol_Fe) ) then
			S(i) = Lw_Fe/ ( cp_p(i)*(Liq_Fe-Sol_Fe) ) * Teil_Fe
			Y_SFe(i) = +S(i) 			
			R_SFe(i) = - S(i) * Sol_Fe			
		else if ( Tp(i) >= Liq_Fe .and. T(i) < Liq_Fe ) then
			S(i) = Lw_Fe/ ( cp_p(i)*(Liq_Fe-Sol_Fe) ) * Teil_Fe
			Y_SFe(i) = +S(i) 			
			R_SFe(i) = - S(i) * Liq_Fe			
		else if ( Tp(i) <= Sol_Fe .and. T(i) > Sol_Fe ) then
			S(i) = Lw_Fe/ ( cp_p(i)*(Liq_Fe-Sol_Fe) ) * Teil_Fe
			Y_SFe(i) = +S(i)/Tp(i)*Sol_Fe 	
			R_SFe(i) = - S(i) * T(i)				
		else if ( Tp(i) <= Liq_Fe .and. T(i) > Liq_Fe ) then
			S(i) = Lw_Fe/ ( cp_p(i)*(Liq_Fe-Sol_Fe) ) * Teil_Fe
			Y_SFe(i) = +S(i)/Tp(i)*Liq_Fe	
			R_SFe(i) = - S(i) * T(i)				
		else
			write(*,*) "Melting error Fe"
			 	stop
		end if
		
		!Melting of silicate phase
		R_SSil(i) = 0.d0
		Y_SSil(i) = 0.d0	
		if (Tp(i) >= Sol_Sil .and. Tp(i) <= Liq_Sil .and. T(i) >= Sol_Sil .and. T(i) <= Liq_Sil) then			
			S(i) = Lw_Sil/ ( cp_p(i)*(Liq_Sil-Sol_Sil) ) * Teil_Sil
			Y_SSil(i) = +S(i) 			
			R_SSil(i) = - S(i) * T(i)				
		else if (Tp(i) < Sol_Sil .and. T(i) < Sol_Sil) then
		else if (Tp(i) > Liq_Sil .and. T(i) > Liq_Sil) then
		else if ( Tp(i) >= Sol_Sil .and. T(i) < Sol_Sil ) then
			S(i) = Lw_Sil/ ( cp_p(i)*(Liq_Sil-Sol_Sil) ) * Teil_Sil
			Y_SSil(i) = +S(i) 			
			R_SSil(i) = - S(i) * Sol_Sil		
		else if ( Tp(i) >= Liq_Sil .and. T(i) < Liq_Sil ) then
			S(i) = Lw_Sil/ ( cp_p(i)*(Liq_Sil-Sol_Sil) ) * Teil_Sil
			Y_SSil(i) = +S(i) 			
			R_SSil(i) = - S(i) * Liq_Sil				
		else if ( Tp(i) <= Sol_Sil .and. T(i) > Sol_Sil ) then
			S(i) = Lw_Sil/ ( cp_p(i)*(Liq_Sil-Sol_Sil) ) * Teil_Sil
			Y_SSil(i) = +S(i)/Tp(i)*Sol_Sil 	
			R_SSil(i) = - S(i) * T(i)			
		else if ( Tp(i) <= Liq_Sil .and. T(i) > Liq_Sil ) then
			S(i) = Lw_Sil/ ( cp_p(i)*(Liq_Sil-Sol_Sil) ) * Teil_Sil
			Y_SSil(i) = +S(i)/Tp(i)*Liq_Sil	
			R_SSil(i) = - S(i) * T(i)			
		else
			write(*,*) "Melting error Sil"
			 	stop
		end if
	
		Y(i) = 1.d0 + dt * ( Br(i)+Bl(i) ) + Y_SFe(i) + Y_SSil(i) 		
		R(i) = - ( T(i)+dt*q(i) ) + R_SFe(i) + R_SSil(i)	
	end do	

	Diskretisierung = Tridia(X,Y,Z,R,k_ein,Zeit) 

	!Stop programme if NaN occurs
	if ((isnan(Diskretisierung(0)) .eqv. .true.) .or. (minval(Diskretisierung) < 0.d0)  )  then
		write(*,*) "isnan in subroutine 'Diskretisierung'"		
		write(*,*) "X",X
		write(*,*) "Y",Y
		write(*,*) "Z",Z
		write(*,*) "R",R		
		write(*,*) "Rp",Rp	
		write(*,*) "Diskretisierung", Diskretisierung
		write(*,*) "dt", dt	
		stop
	end if
end function Diskretisierung

!-----------------------------------------------------------------------------------------------------

!Solving tridiagonal equation system of dimension k_ein
!Input: X,Y,Z,R from function Diskretisierung, number of surface layer k_ein, time Zeit
function Tridia(X,Y,Z,R,k_ein,Zeit)
	use Modelleinstellungen, only: Bildungszeit,T_0,T_S,k
	use Naturkonstanten, only: Z_Scheibe
	use omp_lib
	implicit none
	real(kind=8), intent(in), dimension(:), allocatable :: X,Y,Z,R
	real(kind=8), dimension(0:k_ein):: U,V
	real(kind=8), dimension(0:k) :: Tridia
	real(kind=8), intent(in) :: Zeit
	integer :: i
	integer,intent(in) :: k_ein

	Tridia=0.d0

	!Boundary conditions
	U = 1.d0
	V = 0.d0

	do i = 1,k_ein-1
		U(i) = - X(i) / (Z(i)*U(i-1)+Y(i))
		V(i) = - (R(i)+Z(i)*V(i-1)) / (Z(i)*U(i-1)+Y(i))
	end do

	!At surface temperature is constant = T_0 (Dirichle-boundary)
	!If disk present, surface temperature is T_S
	if (Zeit+Bildungszeit < Z_Scheibe) then
		Tridia(k_ein) = T_S
	else
		Tridia(k_ein) = T_0
	end if

	do i = k_ein-1,0,-1
		Tridia(i) = Tridia(i+1) * U(i) + V(i)
	end do
	if (k_ein<k) then
		do i = k_ein+1,k
			if (Zeit+Bildungszeit < Z_Scheibe) then
				Tridia(i) = T_S
			else
				Tridia(i) = T_0
			end if
		end do 
	end if

	!Tridia, U, and V have to be positive. If they are not an error occurs.
	if ( minval(Tridia) < 0.d0 .or. minval(U)< 0.d0 .or. minval(V)<0.d0 ) then
		write(*,*) "In Tridia"
		write(*,*) "X",X
		write(*,*) "Y",Y
		write(*,*) "Z",Z
		write(*,*) "R",R
		write(*,*) "U",U
		write(*,*) "V",V
		write(*,*) "Tridia        ", Tridia	
		stop
	end if
end function Tridia

!-----------------------------------------------------------------------------------------------------

!Calculation of the sintering process according to theory of Helle et al.
!Input: temperature T, pressure P, gas pressure P_G, filling density D, filling density of new 
!time step D_p, time step size dt, number of surface layer k_ein, initial grain size R0
function Sintern(T,P,P_G,D,D_p,dt,k_ein,R0)							
	use Naturkonstanten
	use Modelleinstellungen
	implicit none	
	real(kind=8), intent(in),dimension(0:k) ::  R0	
	real(kind=8), dimension(:),allocatable, intent(in) :: T,P,P_G,D,D_p
	real(kind=8), dimension(:),allocatable :: Sintern
	real(kind=8), intent(in) :: dt
	real(kind=8) :: A1, P_V, D_pot, D_vol, D_Grenz
	real(kind=8),dimension(0:k) :: R_0	
	real(kind=8),dimension(:),allocatable :: P_eff,Diff_v,mue,deltaD_b, C
	integer :: i
	integer,intent(in) :: k_ein	
	
	allocate(P_eff(0:k))
	allocate(Diff_v(0:k))
	allocate(mue(0:k))
	allocate(deltaD_b(0:k))
	allocate(C(0:k))
	allocate(Sintern(0:k))			
	
	do i=0,k	
		if (chondrendominiert .eqv. .true.) then
			R_0(i) = R0(i)*(1.d0+f_ma)**(drittel)
		else
			R_0(i) = R0(i)
		end if
	end do	

	D_pot = 0.d0;	D_vol = 0.d0;	D_Grenz = 0.d0
	Sintern=D_p
	
	Diff_v = Diff_v0 * exp(-Q_v/R_gSI/T)
	mue = mue_0 * (1.d0+(T-300.d0)/T_m * dmudt )
	deltaD_b = deltaD_0b *exp(-Q_b/R_gSI/T)		
	C = A*b*Diff_v/(k_B*T*mue**(n_Sinter-1))	
	
	A1 = 5.3d0/sqrt(3.d0) * D0_H**(drittel)/sqrt(1.d0-D0_H)
	do i=0,k_ein		
		!checking sintering stage (sintering complete, Pores not closed, Pores closed)
		if (D_p(i) >= D0_H+0.01d0) then
			if (D_p(i) >= 0.999d0) then 
				Sintern(i) = 1.d0									
			else
				!Sintering at high porosities (>10%) after pore closure
				if (D_p(i) < 0.9d0) then						
					P_eff(i) = P(i) * (1.d0-D0_H)/(D_p(i)**2 * (D_p(i)-D0_H) ) + &
						&3.d0*O_Spann/R_0(i)*D_p(i)**2 *(2.d0*D_p(i)-D0_H) / (1.d0-D0_H) + P_g(i)  	
						
					!powerlaw creep					
					D_pot =  A1 * D_p(i)**(zdrittel) * sqrt(D_p(i)-D0_H) * (P_eff(i)*drittel)**n_Sinter	* C(i)
			
					!volume diffusion
					D_vol =  7.d0 * (1.d0-D0_H)/(D_p(i)-D0_H) * Diff_v(i)/R_0(i)**2 * P_eff(i)*Omega/(k_B*T(i))
								
					!surface boundary diffusion
					D_Grenz =  43.d0 * ((1.d0-D0_H)/(D_p(i)-D0_H))**2 * deltaD_b(i)/R_0(i)**3 * P_eff(i)*Omega/(k_B*T(i)) 					

					!Sum up sintering processes
					Sintern(i) = D(i) + (D_pot + D_vol + D_Grenz) * dt*spa
				
				 	if ( Sintern(i) > 1.d0) Sintern(i) = 1.d0				 						
				else if ( D_p(i) >= 0.9d0 .and. D_schliessen(i) .eqv. .true.) then	
					!Sintering at low porosities (<10%)	before pore closure	
					P_V = P_C * (1.d0-D_c)*D_p(i) / ((1.d0-D_p(i))*D_c)
					P_eff(i) = P(i) + 2.d0*O_Spann/R_0(i) * (6.d0*D_p(i)/(1.d0-D_p(i)))**(drittel) - P_V	
					
					!powerlaw creep
					D_pot =  1.5d0* D_p(i)*(1.d0-D_p(i)) / (1.d0-(1.d0-D_p(i))**(n_sinter**(-1)))**n_Sinter &
						&* C(i) * (3.d0*P_eff(i)/n_Sinter*0.5d0)**n_Sinter	
					
					!volume diffusion
					D_vol =  270d0 * 6.d0**(-1.d0/3.d0) * (1.d0-D_p(i))**(5.d0/6.d0) * Diff_v(i)/R_0(i)**2 * P_eff(i)*Omega/(k_B*T(i))
					
					!surface boundary diffusion
					D_Grenz = 270d0* sqrt(1.d0-D_p(i)) * deltaD_b(i)/R_0(i)**3 * P_eff(i)*Omega/(k_B*T(i))
					
					!Sum up sintering processes		
					Sintern(i) =  D(i) + (D_pot + D_vol + D_Grenz) * dt*spa					
					if ( Sintern(i) > 1.d0) Sintern(i) = 1.d0										
				else if (D_p(i) >= 0.9d0 .and. D_schliessen(i) .eqv. .false.) then
					Sintern(i) = 0.90d0
					D_schliessen(i) = .true.					
				else
					!Test if all posibilities are defined
					write(*,*) 'Warning in function Sintern'
					write(*,*) D_p
					write(*,*) D_schliessen
					stop
				end if
			end if
		end if		
	end do
	
	!Program stops if filling density "Sintern" is not 0 <= Sintern <= 1
	if ((maxval(Sintern)>1.d0) .or. (minval(Sintern) <0.d0))then
		write(*,*) "Sintern",Sintern
		write(*,*) "D",D		
		write(*,*) "dt*spa",dt*spa
		stop
	end if	
end function Sintern

!-----------------------------------------------------------------------------------------------------

!Sintering process according to Kakar,Chacklader,Rao
!Input: filling density D, filling density of new time step D_p, temperature T, pressure P
function SinternYom(D,D_p,T,P,dt,k_ein)
	use Naturkonstanten
	use Modelleinstellungen
	implicit none
	integer :: i
	integer,intent(in) :: k_ein
	real(kind=8), intent(in) :: dt
	real(kind=8), dimension(:),allocatable,intent(in) :: D,T,P,D_p
	real(kind=8), dimension(:),allocatable :: R_k
	real(kind=8) :: test, test2, R_0, x
	real(kind=8), dimension(:),allocatable :: D_test, lD,sigma
	real(kind=8), dimension(:),allocatable :: adR2, adR2test, R_test
	real(kind=8), dimension(:),allocatable :: SinternYom
	allocate(SinternYom(0:k))
	allocate(D_test(0:k))
	allocate(lD(0:k))	
	allocate(adR2(0:k))
	allocate(adR2test(0:k))
	allocate(R_test(0:k))
	allocate(sigma(0:k))
	allocate(R_k(0:k))
	
	SinternYom = D

	R_0 = Korn_H*(1.d0/vpid)**(drittel)

	do i=0,k
		if (chondrendominiert .eqv. .true.) then
			R_k(i)= R_0*(1.d0+f_ma)**(drittel)			
		else
			R_k(i) = R_0
		end if
	end do
	
	do i = 0,k_ein
		test = 1.d0;	test2 = 1.d0	
		
		if (D_p(i) >= D0_Y) then !setzt, wann überhaupt gesintert werden soll
			if (D_p(i) <  0.9d0 ) then	!Grenze, ab der der Trick angewandt wird								
				do while ( test2 > 1.d-12)	
					!(a/r)²
					adR2test(i) = adR2(i)			
					adR2(i) = 1.d0 - (0.61d0/D_p(i))**(zdrittel) * (R_0/(R_k(i)+1.d-15))**2	

					!calculation of current radius
					R_test(i) = R_k(i)					
					R_k(i) =  ( R_0**3 / ( ( 2.d0*(2.d0+adR2(i))*sqrt(1.d0-adR2(i)) ) - 3.d0 ))**(drittel)					 
					test2 = abs(adR2test(i)-adR2(i))/(adR2test(i) +1.d-15)			
					if (adR2(i) > 0.688d0) then	
						adR2(i) = 0.6881d0
						R_k(i) =  ( R_0**3 / ( ( 2.d0*(2.d0+adR2(i))*sqrt(1.d0-adR2(i)) ) - 3.d0 ))**(drittel)								
						exit
					end if					
				end do			

				!Calculation of tension
				sigma(i) =  P(i) * 2.d0*sqrt(3.d0)/( pi * ( (4.d0*sqrt(3.d0) * D_p(i) )**(zdrittel) * (R_k(i)/R_0)**2  - 1.d0) ) 			
	
				x = A_S_c * (sigma(i)/1.d5)**n_sigma_S_c/R_k(i)**n_R_S_c * exp(-B_S_c/(R_gSI*T(i))) + &
				&  A_S_d * (sigma(i)/1.d5)**n_sigma_S_d/R_k(i)**n_R_S_d * exp(-B_S_d/(R_gSI*T(i)))
				SinternYom(i) = D(i) +  D_p(i)*x *dt*spa		
			else				 	
				x =  A_S_c * (P(i)/1.d5)**n_sigma_S_c/R_k(i)**n_R_S_c * exp(-B_S_c/(R_gSI*T(i))) + &
				&  A_S_d * (P(i)/1.d5)**n_sigma_S_d/R_k(i)**n_R_S_d * exp(-B_S_d/(R_gSI*T(i))) 
				if (D(i)<1.d0) SinternYom(i) = D(i) + 10.d0*(1.d0-D(i)) * x *dt*spa				
			end if	
		end if		
		if (SinternYom(i) >= 1.d0)	SinternYom(i) = 1.d0	
	end do					
end function SinternYom

!-----------------------------------------------------------------------------------------------------

!Modelling linear growth of the body
!This function increases linearly the mass in the outermost shell. If the outermost shell is too full, a new shell is added by 
!dividing the outermost shell in two, so that the inner shell has the mass it is supposed to have and the outer shell is the new
!outermost shell. The temperature of the new shell is set to the surface temperature.
!Input: Current number of shells k_aktuell, Mass shell distribution M, predetermined mass shell distribution M_Vorgabe, 
!timestep size dt, initial mass M_Anfang, temperature distribution T
function Wachstum(k_aktuell,M,M_Vorgabe,dt,M_Anfang,T)
	use Modelleinstellungen
	implicit none
	integer, intent(inout) :: k_aktuell	
	real(kind=8), dimension(:), allocatable, intent(in) :: M,M_Vorgabe
	real(kind=8), dimension(:), allocatable, intent(inout) :: T
	real(kind=8), intent(in) :: dt,M_Anfang
	real(kind=8), dimension(:),allocatable :: Wachstum
	real(kind=8) :: Einfallrate
	integer :: i	
	allocate(Wachstum(0:k))

	Wachstumsstop = .false.

	!Mass accretion in time step
	Einfallrate = (M_Vorgabe(k)-M_Anfang) * dt/Akkretionsdauer

	Wachstum = M
	Wachstum(k_aktuell) = Wachstum(k_aktuell) + Einfallrate	

	!Generating new shell if outermost shell is to full.
	if ((Wachstum(k_aktuell)-Wachstum(k_aktuell-1)) > 1.5d0*(M_Vorgabe(k_aktuell)-M_Vorgabe(k_aktuell-1)) .and. k_aktuell/= k ) then	
		Wachstum(k_aktuell+1) = Wachstum(k_aktuell)
		Wachstum(k_aktuell) = M_Vorgabe(k_aktuell)		
		
		!Preventing that asteroid grows more than one shell. Just testing if after generation of new 
		!shell the new shell is still too full. Then timestep gets reduced.	
		if ( Einfallrate > 0.2d0*(M_Vorgabe(k_aktuell+1)-M_Vorgabe(k_aktuell)) ) then				
			Wachstumsstop = .true.				
		end if
		
		k_aktuell = k_aktuell + 1
		T(k_aktuell) = T(k_aktuell-1)
	end if
	
	!Adapting Mask to indicate which predetermined shells are in use.
	do i=0,k_aktuell-1
		MaskeT(i) = .true.	
	end do
	
	do while (Wachstum(k_aktuell) >= M_Vorgabe(k) .and. k_aktuell /= k) 
		Wachstum(k_aktuell+1) = Wachstum(k_aktuell)
		Wachstum(k_aktuell) = M_Vorgabe(k_aktuell)
		k_aktuell = k_aktuell+1
		T(k_aktuell) = T(k_aktuell-1)
	end do
	if (Wachstum(k) > M_Vorgabe(k) ) Wachstum(k) = M_vorgabe(k)
end function Wachstum

!-----------------------------------------------------------------------------------------------------

!This function interpolates the input data paires x,y,z at the position "Eingangswert" and has as 
!output the function value of "Eingangswert".
function quadratischeInterpolation(x,y,Eingangswert)	
	implicit none
	real(kind=8) :: quadratischeInterpolation
	real(kind=8),intent(in),dimension(0:2) :: x,y
	real(kind=8),intent(in) :: Eingangswert
	real(kind=8),dimension(0:2) :: l

	l(0) = (Eingangswert-x(1))*(Eingangswert-x(2)) / ((x(0)-x(1))*(x(0)-x(2)))
	l(1) = (Eingangswert-x(0))*(Eingangswert-x(2)) / ((x(1)-x(0))*(x(1)-x(2)))
	l(2) = (Eingangswert-x(0))*(Eingangswert-x(1)) / ((x(2)-x(0))*(x(2)-x(1)))

	quadratischeInterpolation = y(0)*l(0)+y(1)*l(1)+y(2)*l(2)
end function quadratischeInterpolation

!-----------------------------------------------------------------------------------------------------

!This function interpolates the three Input values (x1,y1), (x2,y2), (x3,y3)
!by a quadratic function and returns the position of the extremum. 
!The value at the extremum is "Ausgabe".
!x1<x2<x3 must be fulfilled
function qMax(x,y,Ausgabe)
	implicit none
	real(kind=8) :: qMax
	real(kind=8), dimension(0:2), intent(in) :: x,y
	real(kind=8), intent(out) :: Ausgabe
	real(kind=8) :: a,b,c,Nenner,Zaehler

	!Denominator
	Nenner = (x(0)-x(1))*(x(0)-x(2))*(x(1)-x(2))

	!Numerator of a
	Zaehler = y(0)*(x(1)-x(2)) - y(1)*(x(0)-x(2)) + y(2)*(x(0)-x(1))
	a = Zaehler / Nenner

	!Numerator of b
	Zaehler = -(y(0)*(x(1)**2-x(2)**2) - y(1)*(x(0)**2-x(2)**2) + y(2)*(x(0)**2-x(1)**2))
	b = Zaehler / Nenner

	!Numerator of c
	Zaehler = y(0)*x(1)*x(2)*(x(1)-x(2)) - y(1)*x(0)*x(2)*(x(0)-x(2)) + y(2)*x(0)*x(1)*(x(0)-x(1))
	c =Zaehler / Nenner

	qMax = -b/a *0.5d0
	Ausgabe = a*qMax**2+b*qMax+c
end function qMax
